// Fullscreen Signature Modal Functionality
	document.addEventListener('DOMContentLoaded', function() {
		console.log('DOM loaded, initializing signature modal');

		const modal = document.getElementById('signatureModal');
		const canvas = document.getElementById('signatureCanvas');

		console.log('Modal element:', modal);
		console.log('Canvas element:', canvas);

		if (!modal) {
			console.error('Signature modal not found!');
			return;
		}

		if (!canvas) {
			console.error('Signature canvas not found!');
			return;
		}

		const ctx = canvas.getContext('2d', {
			willReadFrequently: true
		});
		let drawing = false;
		let last = {
			x: 0,
			y: 0
		};

		// Modal controls - handle edit form button only
		function openSignatureModal() {
			modal.classList.add('active');
			document.body.style.overflow = 'hidden';
			setTimeout(() => {
				setCanvasDisplaySize();
			}, 100);
		}



		// Show existing signature preview below button
		function showSignaturePreview() {
			const editSignatureInput = document.getElementById('edit_signature');
			const preview = document.getElementById('edit_signature_preview');
			const img = document.getElementById('edit_signature_img');

			if (editSignatureInput && editSignatureInput.files && editSignatureInput.files.length > 0 && preview && img) {
				// Create URL for the uploaded file
				const fileURL = URL.createObjectURL(editSignatureInput.files[0]);
				img.src = fileURL;
				preview.style.display = 'block';
				console.log('Signature preview shown');
			} else if (preview) {
				preview.style.display = 'none';
				console.log('No signature to preview');
			}
		}

		// Edit form signature button
		const editFormBtn = document.getElementById('openSignatureModalBtnEdit');
		console.log('Edit form button found:', editFormBtn);
		console.log('Modal element found:', modal);

		if (editFormBtn) {
			editFormBtn.addEventListener('click', function() {
				console.log('Button clicked, opening modal');
				openSignatureModal();
			});
		} else {
			console.error('openSignatureModalBtnEdit not found');
		}

		// Show signature preview on page load if signature exists
		showSignaturePreview();
		document.getElementById('closeSignatureModalBtn').addEventListener('click', () => {
			modal.classList.remove('active');
			document.body.style.overflow = 'auto';
		});

		// Close modal on Escape key
		document.addEventListener('keydown', (e) => {
			if (e.key === 'Escape' && modal.classList.contains('active')) {
				modal.classList.remove('active');
				document.body.style.overflow = 'auto';
			}
		});

		// Set canvas to true fullscreen resolution
		function setCanvasDisplaySize() {
			const screenW = window.innerWidth;
			const screenH = window.innerHeight;
			const ratio = window.devicePixelRatio || 1;

			// Set CSS size to full screen
			canvas.style.width = screenW + 'px';
			canvas.style.height = screenH + 'px';

			// Set internal resolution to actual screen resolution * device pixel ratio
			canvas.width = Math.round(screenW * ratio);
			canvas.height = Math.round(screenH * ratio);

			// Scale context for device pixel ratio
			ctx.scale(ratio, ratio);
			ctx.lineCap = 'round';
			ctx.lineJoin = 'round';
			redrawBackground();
		}

		function redrawBackground() {
			// white background for signature
			ctx.save();
			ctx.fillStyle = '#fff';
			ctx.fillRect(0, 0, canvas.width, canvas.height);
			ctx.restore();
		}

		function getPointerPos(evt) {
			const rect = canvas.getBoundingClientRect();
			return {
				x: evt.clientX - rect.left,
				y: evt.clientY - rect.top
			};
		}

		// Drawing event listeners
		canvas.addEventListener('pointerdown', (e) => {
			e.preventDefault();
			canvas.setPointerCapture(e.pointerId);
			drawing = true;
			last = getPointerPos(e);
		});

		canvas.addEventListener('pointermove', (e) => {
			if (!drawing) return;
			const p = getPointerPos(e);
			const penWidthSelect = document.getElementById('penWidth');
			const penColorSelect = document.getElementById('penColor');

			ctx.strokeStyle = penColorSelect.value;
			ctx.lineWidth = Number(penWidthSelect.value);
			ctx.beginPath();
			ctx.moveTo(last.x, last.y);
			ctx.lineTo(p.x, p.y);
			ctx.stroke();
			last = p;
		});

		canvas.addEventListener('pointerup', (e) => {
			drawing = false;
			canvas.releasePointerCapture(e.pointerId);
		});

		// Prevent scrolling while drawing on touch devices
		canvas.addEventListener('touchstart', e => e.preventDefault(), {
			passive: false
		});

		// Control buttons
		document.getElementById('clearSignatureBtn').addEventListener('click', () => {
			ctx.clearRect(0, 0, canvas.width, canvas.height);
			redrawBackground();
		});

		document.getElementById('saveSignatureBtn').addEventListener('click', async () => {
			// Convert canvas to data URL
			const dataURL = canvas.toDataURL('image/png');

			try {
				// Convert base64 to blob
				const response = await fetch(dataURL);
				const blob = await response.blob();

				// Create a File object from the blob
				const timestamp = new Date().getTime();
				const fileName = `signature_${timestamp}.png`;
				const file = new File([blob], fileName, {
					type: 'image/png'
				});

				// Create a DataTransfer object to populate the file input
				const dataTransfer = new DataTransfer();
				dataTransfer.items.add(file);

				// Get the file input and assign the file
				const fileInput = document.getElementById('edit_signature');
				if (fileInput) {
					fileInput.files = dataTransfer.files;

					// Trigger change event if needed for form validation
					const event = new Event('change', {
						bubbles: true
					});
					fileInput.dispatchEvent(event);

					console.log('Signature file uploaded automatically:', fileName);

					// Show preview in edit form
					const preview = document.getElementById('edit_signature_preview');
					const img = document.getElementById('edit_signature_img');
					if (preview && img) {
						img.src = dataURL;
						preview.style.display = 'block';
					}
				}

				// Close modal
				modal.classList.remove('active');
				document.body.style.overflow = 'auto';

				// alert('Signature saved successfully!');
			} catch (error) {
				console.error('Error saving signature:', error);
				// alert('Error saving signature. Please try again.');
			}
		});

		// Clear saved signature for edit form
		const clearEditSignatureBtn = document.getElementById('clearEditSignature');
		if (clearEditSignatureBtn) {
			clearEditSignatureBtn.addEventListener('click', () => {
				const editSignatureInput = document.getElementById('edit_signature');
				if (editSignatureInput) {
					// Clear the file input by recreating it
					const newInput = document.createElement('input');
					newInput.type = 'file';
					newInput.id = 'edit_signature';
					newInput.name = 'edit_signature';
					newInput.accept = 'image/*';
					newInput.style.display = 'none';
					editSignatureInput.parentNode.replaceChild(newInput, editSignatureInput);
					showSignaturePreview(); // Update preview after clearing
				}
			});
		}

		window.addEventListener('resize', () => {
			if (modal.classList.contains('active')) {
				setCanvasDisplaySize();
			}
		});
	}); // End of DOMContentLoaded