<?php
// Set JSON content type and CORS headers
header("Content-Type: application/json");
header("Access-Control-Allow-Origin: *");
header("Access-Control-Allow-Methods: POST");
header("Access-Control-Allow-Headers: Access-Control-Allow-Headers, Content-Type, Access-Control-Allow-Methods, Authorization, X-Requested-With");
require_once "db_connect.php";

$sys_res = $db->query("SELECT * FROM system_settings");
$sysinfo = $sys_res->fetch_assoc();
date_default_timezone_set($sysinfo['system_time_zone']);

// Function to get mappings from database
function getMappings($db)
{
    // First get all distinct instrument types from the database
    $types_query = $db->query("SELECT DISTINCT instrument_type FROM lab_devices WHERE is_active = 1");
    $instrument_types = [];
    while ($row = $types_query->fetch_assoc()) {
        $instrument_types[] = $row['instrument_type'];
    }

    // Then get all mappings for these instrument types
    $stmt = $db->query("SELECT instrument_type, test_name, test_code, is_detail 
                        FROM lab_devices 
                        WHERE is_active = 1
                        ORDER BY instrument_type, is_detail");

    $result = [];

    // Initialize all mappings dynamically based on what's in the database
    foreach ($instrument_types as $type) {
        $result["{$type}_test_map"] = [];
        $result["{$type}_test_detail_map"] = [];
    }

    while ($row = $stmt->fetch_assoc()) {
        if ($row['is_detail']) {
            $varName = $row['instrument_type'] . '_test_detail_map';
        } else {
            $varName = $row['instrument_type'] . '_test_map';
        }

        // Check if test_name already exists in the array
        if (isset($result[$varName][$row['test_name']])) {
            // If it exists and is not already an array, convert it to an array
            if (!is_array($result[$varName][$row['test_name']])) {
                $result[$varName][$row['test_name']] = [$result[$varName][$row['test_name']]];
            }
            // Add the new test_code to the array
            $result[$varName][$row['test_name']][] = $row['test_code'];
        } else {
            // First occurrence, store as single value
            $result[$varName][$row['test_name']] = $row['test_code'];
        }
    }
    return $result;
}

// Function to return API responses
function sendResponse($success, $data, $httpCode = 200)
{
    http_response_code($httpCode);
    echo json_encode([
        'success' => $success,
        'data' => $data,
        'timestamp' => date("Y-m-d H:i:s")
    ]);
    exit;
}

// Function to authenticate user
function authenticateUser($username, $password)
{
    global $db;

    $password = stripslashes($password);
    $password = mysqli_real_escape_string($db, $password);

    $stmt = $db->prepare("SELECT staff_id, username, password FROM staff WHERE username = ? AND is_deleted = 0");
    $stmt->bind_param("s", $username);
    $stmt->execute();
    $result = $stmt->get_result();

    if ($result->num_rows === 1) {
        $user = $result->fetch_assoc();
        if (password_verify($password, $user['password'])) {
            return [
                'authenticated' => true,
                'user_id' => $user['staff_id'],
                'username' => $user['username'],
            ];
        }
    }

    return ['authenticated' => false];
}

function flippedMap($test_map)
{
    $flipped_map = [];
    foreach ($test_map as $name => $codes) {
        // Handle both single codes and arrays of codes
        if (is_array($codes)) {
            foreach ($codes as $code) {
                if (!isset($flipped_map[$code])) {
                    $flipped_map[$code] = [];
                }
                $flipped_map[$code][] = $name;
            }
        } else {
            if (!isset($flipped_map[$codes])) {
                $flipped_map[$codes] = [];
            }
            $flipped_map[$codes][] = $name;
        }
    }
    return $flipped_map;
}

function getResult($test_id, $test_mapping, $result)
{
    if (isset($test_mapping[$test_id])) {
        $test_names = $test_mapping[$test_id];
        if (is_array($test_names)) {
            foreach ($test_names as $name) {
                if (isset($result[$name])) {
                    return $result[$name];
                }
            }
        } elseif (isset($result[$test_names])) {
            return $result[$test_names];
        }
    }
    return '';
}

$requestData = [];
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    // Check if we have form data first
    if (!empty($_POST)) {
        $requestData = $_POST;
    } else {
        // If no form data, try to parse JSON
        $jsonInput = file_get_contents('php://input');
        if (!empty($jsonInput)) {
            $requestData = json_decode($jsonInput, true) ?? [];
        }
    }
} else {
    // Parse JSON input for other methods
    $jsonInput = file_get_contents('php://input');
    if (!empty($jsonInput)) {
        $requestData = json_decode($jsonInput, true) ?? [];
    }
}

// Check for credentials in request
if (!isset($requestData['username']) || !isset($requestData['password'])) {
    sendResponse(false, "Username and password required", 401);
}

// Authenticate user
$auth = authenticateUser($requestData['username'], $requestData['password']);
if (!$auth['authenticated']) {
    sendResponse(false, "Invalid username or password", 401);
}

// Set user information
$user_id = $auth['user_id'];
$username = $auth['username'];

// Handle API operations
if (!isset($requestData['device'])) {
    sendResponse(false, "Device not specified", 404);
}

// After getting the device name from request:
$device = strtolower(mysqli_real_escape_string($db, $requestData['device']));

// Get all active instrument types from database
$types_query = $db->query("SELECT DISTINCT instrument_type FROM lab_devices WHERE is_active = 1");
$valid_instrument_types = [];
while ($row = $types_query->fetch_assoc()) {
    $valid_instrument_types[] = $row['instrument_type'];
}

// Directly use the device name as instrument type after validation
if (!in_array($device, $valid_instrument_types)) {
    sendResponse(false, "Device not registered or not active", 404);
}

$instrument_type = $device;

// Get all test mappings from database
$mappings = getMappings($db);
$test_map = $mappings["{$instrument_type}_test_map"] ?? [];
$test_detail_map = $mappings["{$instrument_type}_test_detail_map"] ?? [];
// sendResponse(true, [
//     'test_map' => $test_map,
//     'test_detail_map' => $test_detail_map
// ]);
// Get lab test ID
$lid = $requestData['lab_id'];
$labquery = $db->query("SELECT L.lab_test_id as lab_id from lab_test as L
                        WHERE L.is_deleted = 0 AND L.lab_test_id ='$lid'
                        ORDER BY L.lab_test_date DESC
                        LIMIT 1");

$id = 0;
$data = $labquery->fetch_all(MYSQLI_ASSOC);
foreach ($data as $row) {
    $id = $row['lab_id'];
}

if ($id === 0) {
    sendResponse(false, "Lab test id not found", 404);
}

// Prepare common update queries
$reg_date = date("Y-m-d H:i:s", time());
$lastaction = "Update Test information by Lab Device";
$by = $user_id;
$srepo = " sample_reported_on='$reg_date', ";

$update_qry = "UPDATE lab_test SET 
                $srepo
                last_action ='$lastaction',
                updated_at ='$reg_date',
                last_action_by ='$by' 
                WHERE lab_test_id = $id";

$update_lab_test_detail = "UPDATE lab_test_detail SET 
                last_action ='$lastaction',
                updated_at ='$reg_date',
                last_action_by ='$by' 
                WHERE lab_test_id_fk = $id";

$update_lab_test_detail_item = "UPDATE lab_test_detail_item SET 
                last_action ='$lastaction',
                updated_at ='$reg_date',
                last_action_by ='$by' 
                WHERE lab_test_id_fk = $id";

// Process result data
$result = $requestData['result'];
if (is_string($result)) {
    $result = json_decode($result, true);
    if (json_last_error() !== JSON_ERROR_NONE) {
        sendResponse(false, "Invalid JSON format in result data", 400);
    }
}

// Get lab test data
$lab_test_items_query = "SELECT LDT.lab_test_detail_item_id, L.lab_test_id, PD.product_id_fk, 
                         P.product_name, PD.product_detail_id as test_id, 
                         PD.product_detail_name as test_name, LDT.item_test_result 
                         FROM lab_test as L
                         LEFT JOIN lab_test_detail_item as LDT ON L.lab_test_id = LDT.lab_test_id_fk 
                         LEFT JOIN product_detail as PD ON PD.product_detail_id = LDT.product_detail_id_fk
                         LEFT JOIN product as P ON P.product_id = PD.product_id_fk
                         WHERE L.lab_test_id = $id order by test_id";

$lab_test_query = "SELECT LD.lab_test_detail_id, P.product_id as test_id, L.lab_test_id, 
                   P.product_name as test_name, LD.test_result
                   FROM lab_test as L
                   LEFT JOIN lab_test_detail as LD ON L.lab_test_id = LD.lab_test_id_fk 
                   LEFT JOIN product as P ON P.product_id = LD.product_id_fk
                   WHERE L.lab_test_id = $id";

// Start transaction
$before_update = $db->query("SELECT * FROM lab_test WHERE lab_test_id=$id")->fetch_assoc();
$db->autocommit(FALSE);

// Execute base updates
if (!$db->query($update_qry) || !$db->query($update_lab_test_detail) || !$db->query($update_lab_test_detail_item)) {
    $db->rollback();
    sendResponse(false, "Error: Update Failed", 400);
}

// Process test results based on device type
$test_mapping = flippedMap($test_map);
$test_detail_mapping = flippedMap($test_detail_map);

// For Medonic, Beckman and Lyte devices, we only update detail items
if ($instrument_type === 'medonic' || $instrument_type === 'beckman' || $instrument_type === 'lyte') {
    $lab_test_items = $db->query($lab_test_items_query)->fetch_all(MYSQLI_ASSOC);
    foreach ($lab_test_items as $item) {
        $item_test_result = getResult($item['test_id'], $test_detail_mapping, $result);
        if ($item_test_result !== '' && $item['lab_test_detail_item_id'] > 0) {
            $update_qry = "UPDATE lab_test_detail_item SET 
                            item_test_result ='$item_test_result',
                            item_test_note ='',
                            last_action ='$lastaction',
                            updated_at ='$reg_date',
                            last_action_by ='$by' 
                            WHERE lab_test_detail_item_id = {$item['lab_test_detail_item_id']}";


            if (!$db->query($update_qry)) {
                $db->rollback();
                sendResponse(false, "Error: Update Failed for test item", 400);
            }
        }
    }
}
// For other devices, we update both main tests and detail items
else {
    // Update main tests
    $lab_tests = $db->query($lab_test_query)->fetch_all(MYSQLI_ASSOC);
    foreach ($lab_tests as $test) {
        $test_id = ($instrument_type === 'cobas' && $test['test_id'] == 1982) ? 30 : $test['test_id'];
        $test_result = getResult($test_id, $test_mapping, $result);

        if ($test_result !== '' && $test['lab_test_detail_id'] > 0) {
            $update_qry = "UPDATE lab_test_detail SET 
                            test_result ='$test_result',
                            test_note ='',
                            last_action ='$lastaction',
                            updated_at ='$reg_date',
                            last_action_by ='$by' 
                            WHERE lab_test_detail_id = {$test['lab_test_detail_id']}";

            if (!$db->query($update_qry)) {
                $db->rollback();
                sendResponse(false, "Error: Update Failed for main test", 400);
            }
        }
    }

    // Update detail items if they exist
    $lab_test_items = $db->query($lab_test_items_query)->fetch_all(MYSQLI_ASSOC);
    foreach ($lab_test_items as $item) {
        $item_test_result = getResult($item['test_id'], $test_detail_mapping, $result);
        if ($item_test_result !== '' && $item['lab_test_detail_item_id'] > 0) {
            $update_qry = "UPDATE lab_test_detail_item SET 
                            item_test_result ='$item_test_result',
                            item_test_note ='',
                            last_action ='$lastaction',
                            updated_at ='$reg_date',
                            last_action_by ='$by' 
                            WHERE lab_test_detail_item_id = {$item['lab_test_detail_item_id']}";

            if (!$db->query($update_qry)) {
                $db->rollback();
                sendResponse(false, "Error: Update Failed for test item", 400);
            }
        }
    }
}

// Commit transaction and send success response
$after_update = $db->query("SELECT * FROM lab_test WHERE lab_test_id=$id")->fetch_assoc();
$db->commit();
sendResponse(true, "Record Successfully Updated", 200);
